/*
 * File:   main.c
 * Author: Tim Blythman
 * Project: SoundFX V2
 *          WAV files from SD card mixed and output to CS4334 DAC
 *          SPI2 used as I2S to DAC
 *          SPI1 for SD Card (CD on RB13, pin24)
 *          RA4 indicator LED
 *          RA2 REG1 Enable
 *          RB6 SW1
 *          RB7 SW2
 *          RC9 SW3
 *          RA3 SW4
 *          RB4 SW5
 *          RB10 SW6
 *          RB11 SW7
 *          RB5 IC3 (amplifier) enable
 * 
 *          configured by config.txt file on card
 * 
 * Created on 12 June 2018, 9:57 AM
 */
void goPower(char p);     //0 for sleep/low, 1 for high/running

#define BUF_COUNT 4
#define DAC_BUF_SIZE 512
#define RAW_BUF_SIZE (DAC_BUF_SIZE*6)
#define CYCLES_PER_SECOND (12000000L)

#include "config.h"
#include <xc.h>
#include "sinetable.h"
#include "samples.h"
//const unsigned char* sampleData[];            //dummies to comment out samples.h
//const uint32_t sampleSize[];
#include "io.h"
#include "sdcard/spi.h"
#include "sdcard/sd.h"
#include "sdcard/ff.h"
#include "util.h"
#include "WAV.h"

//global settings changed by config file
uint32_t powerTimeOut=0;                        //default value is off
uint32_t powerTimeOutCheck=0;                   //time of last activity
int pStat=-1;                                   //used by goPower, init status unknown
int32_t masterVolume=256;                        // 0-256
char mainMode=0;                                 //see MAINMODE in play.h
char speakBack=2;                                //default to all
char SW4mode=0;                                  //default to normal trigger, see SW4MODE in play.h
uint32_t SW4rawvalue=0;                          //derived from analog/frequency input, will be zero when SW4 is trigger
uint32_t SW4scalevalue=0;                        //adjusted from SW4rawvalue to 0-100% (constrained to range), will be zero when SW4 is trigger
uint32_t SW4rawlow=0;                            //what raw value scales to 0%, read from card
uint32_t SW4rawhigh=0;                           //what raw value scales to 100%, read from card
#define SW4_FULL_SCALE 256
extern uint32_t speedScaleMin[];                //scaling that applies when SW4scalevalue=0, loaded from card, see play.h, applied per input
extern uint32_t speedScaleMax[];                //scaling that applies when SW4scalevalue=SW4_FULL_SCALE, loaded from card, see play.h
extern uint32_t speedScaleMinChannel[];         //scaling that applies when SW4scalevalue=0, set at playback
extern uint32_t speedScaleMaxChannel[];         //scaling that applies when SW4scalevalue=SW4_FULL_SCALE, set at playback
#define SPEED_SCALE_DIVIDER 256
char reg1Needed=1;                               // can be set to zero if 12V supply is detected/set, default to yes, doesn't make much difference to power use
//char reg1Needed=0;                               // can be set to zero if 12V supply is detected/set, default to yes

unsigned char memoryCardSystemUp=0;              // flag indicates whether SD card has been detected & configured
extern CARD_INFO cardInfo;
static FATFS fs;
char playing[BUF_COUNT];                         // set to zero on finish playback
char playingInt[BUF_COUNT];                      //playing from internal samples
char fileToRead[BUF_COUNT];                      // data available to read
char clipping=0;                                 //set if clipping detected
char activeLogicLevel=0;                         //0 is active low (normal), 1 is high

uint16_t stat;
uint16_t d=0,i=0;
char rootbuf[5] = "0:\\";
const char* pbuf;    
FATFS* pfs;
DIR dd;
FRESULT f[BUF_COUNT];
FIL filetoopen[BUF_COUNT];

//parameters for mixed mode operation
#define INPUT_COUNT 7
#define HANDLES_PER_INPUT 10
FIL mixedhandles[INPUT_COUNT][HANDLES_PER_INPUT];     //7 inputs, up to HANDLES_PER_INPUT handles each for mixed mode
//int32_t soundVolume[INPUT_COUNT];                     // 0-256
char soundMode[INPUT_COUNT];                          //see MIXEDINPUT enum in play.h
char bufInUse[INPUT_COUNT];                           //which buffer is the trigger using?
char inputPhase[INPUT_COUNT];                         //progress of sound sequencing

//for storing DAC data
int16_t dacbuf[BUF_COUNT][3][DAC_BUF_SIZE];             //data ready for DAC
volatile unsigned int dacbufinuse[BUF_COUNT];                    //which buffer of 3 in use
volatile unsigned int dacbufptr[BUF_COUNT];                      //pointer in that buffer
volatile unsigned int dacbufend[BUF_COUNT][3];                   //number of samples available in buffer
signed int dacbufVolume[BUF_COUNT];                     //volume for current buffer, set from input

signed int switchVolumeMin[INPUT_COUNT];                 //for adjustable volumes
signed int switchVolumeMax[INPUT_COUNT];
signed int dacbufVolumeMin[BUF_COUNT];
signed int dacbufVolumeMax[BUF_COUNT];

//wav info
unsigned long wav_sample_rate[BUF_COUNT];
unsigned char wav_num_channels[BUF_COUNT];
unsigned char wav_bytes_per_sample[BUF_COUNT];
int fBuf=-1;                                            //find free buffer

//parameters for select from card mode, 64 options based on 6 binary selectors
//#define SELECT_OPTIONS 64
//FIL selectHandles[SELECT_OPTIONS];              //each handle is 36 bytes

#include "play.h"
volatile uint32_t inputFreq;             //for main to read

void main(void) {
    unsigned int k,b;               //counters  
    for(b=0;b<BUF_COUNT;b++){       //initialise buffer variables
        playing[b]=0;
        playingInt[b]=0;
        fileToRead[b]=0;
        dacbufinuse[b]=0;
        dacbufptr[b]=0;
        dacbufVolume[b]=256;        //default in case not set
        dacbufVolumeMin[b]=256;     //default
        dacbufVolumeMax[b]=256;
    }
    for(b=0;b<INPUT_COUNT;b++){     //set up triggers
        soundMode[b]=0;                                       //nothing
        bufInUse[b]=0;                                        //which buffer is the trigger using?
        inputPhase[b]=0;
        speedScaleMin[b]=SPEED_SCALE_DIVIDER;                   //default to no adjustment
        speedScaleMax[b]=SPEED_SCALE_DIVIDER;                   //default to no adjustment
        switchVolumeMin[b]=256;                                 //for adjustable volumes
        switchVolumeMax[b]=256;
    }
    pbuf = rootbuf;
	for(k=0;k<BUF_COUNT;k++){WAVinitHeader(&header[k]);}

    IOInit();                                   //ensure boost reg is shut down initially
    reg1Set(reg1Needed);                        //5V rail for DAC/AMP
    if(PORTBbits.RB13!=0){                      //no card =>
        IOInit();                               //ensure boost reg is shut down initially    
        ampSet(1);                              //turn amp on
        reg1Set(reg1Needed);                    //5V rail for DAC/AMP
        playSampleSelect();                     //play internal sample until card detected
        reg1Set(0);                             //
        ampSet(0);                              //turn amp off
        delay(100);        
    }
    memoryCardSystemUp=1;                       //card detected    
   
    for(i=0;i<10;i++){                          //test for FS without REG1
        f_mount(0, &fs);                        //mount fs
        stat=auto_mount(&pbuf, &pfs, 0);        //inits fs and spi etc
        if(stat==FR_OK){break;}
    }
    if(stat){errorflash(stat);}     //FS error
    
    //otherwise, file system is up
    if(!loadConfig("CONFIG.TXT")){errorflash(5);}    //load config file and set operation mode  

    ampSet(1);                                  //turn amp on
    delay(100);
    IEC1bits.SPI2TXIE=1;                        //enable TX interrupt, needed for any audio output
    
    LEDSet(1);
    if(speakBack){                          // not none
        if(speakBack==1){
            speakConfig(0);                 //summary info
        }else{
            speakConfig(INPUT_COUNT);       //all info            
        }
    }
    LEDSet(0);    
    goPower(1);                             //ensure evrything is up
    srand(_CP0_GET_COUNT());                //randomize, delays in SD startup should give enough jitter
    powerTimeOutCheck=_CP0_GET_COUNT();
    while(1){                               // Main loop
        k=0;
        for(b=0;b<BUF_COUNT;b++){           //check if anything is playing
            if(playing[b]){k=1;}
            if(playingInt[b]){k=1;}
        }
        if(k){
            LEDSet(!clipping);              //LED goes out on clipping,
            powerTimeOutCheck=_CP0_GET_COUNT(); //reset counter if playback occurring
        }else{
            LEDSet(0);                      //stays off till playback starts next
            clipping=0;            
            if(powerTimeOut){               //if a timeout has been set
                if((_CP0_GET_COUNT()-powerTimeOutCheck)>(powerTimeOut*CYCLES_PER_SECOND)){
                    powerTimeOutCheck=_CP0_GET_COUNT(); //reset counter to prevent repeated shutDown asserts
                    goPower(0);             //shutdown everything                    
                }
                if(!pStat){       //sleep if suitable (try while loop testing outputs to avoid full main cycle between sleeps)
                    while(1){
                        sleep(1);
                        for(k=0;k<INPUT_COUNT;k++){
                            if(SW4mode&&(k==3)){k++;}               //skip SW4 if it's doing speed control        
                            if(getSW(k+1)==activeLogicLevel){k=99;} //flag it, simple break comes out of for instead of while
                        }
                        if(k>90){break;}
                    }
                }
            }
        }   
        for(b=0;b<BUF_COUNT;b++){           //check buffers
            if(playing[b]){
                if(dacbufend[b][0]+dacbufend[b][1]+dacbufend[b][2]==0){
                    playing[b]=0;
                }
                for(k=0;k<3;k++){                                  //may need to flag only one load per cycle
                   if((dacbufend[b][k]==0)&&(fileToRead[b])){      //buffer empty
                        if(getDACsamples(b,k)==0){                 //try to top up buffer
                            fileToRead[b]=0;                       //no more data
                        }else{
                            //break;                                  //only load one buffer per loop
                        }
                    }
                }
            }
        }
        for(b=0;b<INPUT_COUNT;b++){                 //check inputs
            if(SW4mode&&(b==3)){b++;}               //skip SW4 if it's doing speed control        
            switch(soundMode[b]){
                case MIXED_LOOP:                                                //loop
                    if((inputPhase[b]==0)&&(getSW(b+1)==activeLogicLevel)){    //idle and button pressed
                        goPower(1);                             //awake everything    
                        fBuf=findBuffer();                      //find free buffer  
                        if(fBuf>=0){                            //whatever buffer is free
                            filetoopen[fBuf]=mixedhandles[b][0];                            
                            openFile(fBuf);                            
                            bufInUse[b]=fBuf;
                            speedScaleMinChannel[fBuf]=speedScaleMin[b];
                            speedScaleMaxChannel[fBuf]=speedScaleMax[b];
                            inputPhase[b]=1;
                            dacbufVolumeMin[fBuf]=switchVolumeMin[b];   //allocate volume
                            dacbufVolumeMax[fBuf]=switchVolumeMax[b];   //allocate volume
                        }
                    }
                    if((inputPhase[b]!=0)&&(playing[bufInUse[b]]==0)){inputPhase[b]=0;}     //reset for next
                    break;
                case MIXED_SINGLE:                                              //play once
                    if((inputPhase[b]==0)&&(getSW(b+1)==activeLogicLevel)){    //idle and button pressed
                        goPower(1);                             //awake everything    
                        fBuf=findBuffer();                      //find free buffer  
                        if(fBuf>=0){
                            filetoopen[fBuf]=mixedhandles[b][0];
                            openFile(fBuf);
                            bufInUse[b]=fBuf;                            
                            speedScaleMinChannel[fBuf]=speedScaleMin[b];
                            speedScaleMaxChannel[fBuf]=speedScaleMax[b];
                            inputPhase[b]=1;
                            dacbufVolumeMin[fBuf]=switchVolumeMin[b];   //allocate volume
                            dacbufVolumeMax[fBuf]=switchVolumeMax[b];   //allocate volume
                        }
                    }
                    if((inputPhase[b]==1)&&(getSW(b+1)!=activeLogicLevel)){inputPhase[b]=2;}               //wait for button release
                    if((inputPhase[b]==2)&&(playing[bufInUse[b]]==0)){inputPhase[b]=0;}     //reset for next
                    break;
                case MIXED_CROPLOOP:                                            //cropped loop (stop if button released)
                    if((inputPhase[b]==0)&&(getSW(b+1)==activeLogicLevel)){    //idle and button pressed
                        goPower(1);                             //awake everything    
                        fBuf=findBuffer();                      //find free buffer  
                        if(fBuf>=0){
                            filetoopen[fBuf]=mixedhandles[b][0];
                            openFile(fBuf);
                            bufInUse[b]=fBuf;                            
                            speedScaleMinChannel[fBuf]=speedScaleMin[b];
                            speedScaleMaxChannel[fBuf]=speedScaleMax[b];
                            inputPhase[b]=1;
                            dacbufVolumeMin[fBuf]=switchVolumeMin[b];   //allocate volume
                            dacbufVolumeMax[fBuf]=switchVolumeMax[b];   //allocate volume
                        }
                    }
                    if((inputPhase[b]!=0)&&(playing[bufInUse[b]]==0)){inputPhase[b]=0;}     //reset for next loop
                    if((inputPhase[b]==1)&&(getSW(b+1)!=activeLogicLevel)){        //stop if button released while playing
                        inputPhase[b]=0;
                        fileToRead[bufInUse[b]]=0;
                        playing[bufInUse[b]]=0;                     //cancel playback
                    }
                    break;
                case MIXED_CROPSINGLE:                                          //cropped single (stop if button released)
                    if((inputPhase[b]==0)&&(getSW(b+1)==activeLogicLevel)){    //idle and button pressed
                        goPower(1);                             //awake everything    
                        fBuf=findBuffer();                      //find free buffer  
                        if(fBuf>=0){
                            filetoopen[fBuf]=mixedhandles[b][0];
                            openFile(fBuf);
                            bufInUse[b]=fBuf;                            
                            speedScaleMinChannel[fBuf]=speedScaleMin[b];
                            speedScaleMaxChannel[fBuf]=speedScaleMax[b];
                            inputPhase[b]=1;
                            dacbufVolumeMin[fBuf]=switchVolumeMin[b];   //allocate volume
                            dacbufVolumeMax[fBuf]=switchVolumeMax[b];   //allocate volume
                        }
                    }
                    if((inputPhase[b]==1)&&(getSW(b+1)!=activeLogicLevel)){        //stop if button released while playing
                        inputPhase[b]=0;
                        fileToRead[bufInUse[b]]=0;
                        playing[bufInUse[b]]=0;                     //cancel playback
                    }
                    break;
                case MIXED_ASR:                                                 //ASR mode 1 {[2]} 3
                    switch(inputPhase[b]){
                        case 0:
                            if(getSW(b+1)==activeLogicLevel){                                  //idle and button pressed
                                goPower(1);                                     //awake everything    
                                fBuf=findBuffer();                              //find free buffer  
                                if(fBuf>=0){
                                    filetoopen[fBuf]=mixedhandles[b][0];
                                    openFile(fBuf);
                                    bufInUse[b]=fBuf;                            
                                    speedScaleMinChannel[fBuf]=speedScaleMin[b];
                                    speedScaleMaxChannel[fBuf]=speedScaleMax[b];
                                    inputPhase[b]=1;
                                    dacbufVolumeMin[fBuf]=switchVolumeMin[b];   //allocate volume
                                    dacbufVolumeMax[fBuf]=switchVolumeMax[b];   //allocate volume
                                }                                
                            }
                            break;    
                        case 1:                                                     //first buffer finished
                            if((fileToRead[bufInUse[b]]==0)||(playing[bufInUse[b]]==0)){
                                if(getSW(b+1)==activeLogicLevel){                                  //button still in
                                    filetoopen[bufInUse[b]]=mixedhandles[b][1];     //play second file, stay in phase 1
                                    openFile(bufInUse[b]);
                                }else{
                                    filetoopen[bufInUse[b]]=mixedhandles[b][2];     //play third file, go to phase 2
                                    openFile(bufInUse[b]);
                                    inputPhase[b]=2;
                                }
                            }
                            break;
                        case 2:
                            if(playing[bufInUse[b]]==0){                        //third file finished, end
                                inputPhase[b]=0;
                            }
                            break;                            
                    }
                    break;
                case MIXED_ALTLOOP:                                             //altloop 1 when pressed, 2 when released
                    if(inputPhase[b]==0){                                       //idle
                        goPower(1);                                             //awake everything    
                        fBuf=findBuffer();                                      //find free buffer  
                        if(fBuf>=0){                                            
                            if(getSW(b+1)==activeLogicLevel){                                  //button pressed                                
                                filetoopen[fBuf]=mixedhandles[b][0];            //play 1
                            }else{
                                filetoopen[fBuf]=mixedhandles[b][1];            //else play 2
                            }
                            openFile(fBuf);                            
                            bufInUse[b]=fBuf;
                            speedScaleMinChannel[fBuf]=speedScaleMin[b];
                            speedScaleMaxChannel[fBuf]=speedScaleMax[b];
                            inputPhase[b]=1;                                    //set flag
                            dacbufVolumeMin[fBuf]=switchVolumeMin[b];   //allocate volume
                            dacbufVolumeMax[fBuf]=switchVolumeMax[b];   //allocate volume
                        }
                    }
                    if((inputPhase[b]!=0)&&(playing[bufInUse[b]]==0)){inputPhase[b]=0;}     //reset for next
                    break;
                case MIXED_ROUNDROBIN:                                          //play through sounds in list in order
                    if((inputPhase[b]&1)==0){                                   //keep track of # through inputPhase
                        if(getSW(b+1)==activeLogicLevel){                                      //button pressed                                
                            goPower(1);                                         //awake everything    
                            fBuf=findBuffer();                                  //find free buffer  
                            if(fBuf>=0){                                            
                                filetoopen[fBuf]=mixedhandles[b][inputPhase[b]/2];
                                if(filetoopen[fBuf].fsize>43){                  //valid file
                                    openFile(fBuf);                            
                                    bufInUse[b]=fBuf;
                                    speedScaleMinChannel[fBuf]=speedScaleMin[b];
                                    speedScaleMaxChannel[fBuf]=speedScaleMax[b];
                                    dacbufVolumeMin[fBuf]=switchVolumeMin[b];   //allocate volume
                                    dacbufVolumeMax[fBuf]=switchVolumeMax[b];   //allocate volume
                                    inputPhase[b]++;                            //set flag
                                }else{
                                    inputPhase[b]=inputPhase[b]+2;              //not valid, jump to next file, processed next loop of main
                                }                                        
                            }
                        }
                    }else{
                        if((getSW(b+1)!=activeLogicLevel)&&(playing[bufInUse[b]]==0)){         //button released and playback finished                               
                            inputPhase[b]++;                                    //get ready for next
                        }
                    }
                    if(inputPhase[b]>=(HANDLES_PER_INPUT*2)){inputPhase[b]=inputPhase[b]-(HANDLES_PER_INPUT*2);}    //loop around
                    break;
                case MIXED_RANDOM:                                              //play sounds in list randomly
                    if(inputPhase[b]==0){                                       //keep track of state
                        if(getSW(b+1)==activeLogicLevel){                                      //button pressed                                
                            goPower(1);                                         //awake everything    
                            fBuf=findBuffer();                                  //find free buffer  
                            if(fBuf>=0){                                            
                                filetoopen[fBuf]=mixedhandles[b][rand()%HANDLES_PER_INPUT];  //random file
                                if(filetoopen[fBuf].fsize>43){                  //valid file
                                    openFile(fBuf);                            
                                    bufInUse[b]=fBuf;
                                    speedScaleMinChannel[fBuf]=speedScaleMin[b];
                                    speedScaleMaxChannel[fBuf]=speedScaleMax[b];
                                    dacbufVolumeMin[fBuf]=switchVolumeMin[b];   //allocate volume
                                    dacbufVolumeMax[fBuf]=switchVolumeMax[b];   //allocate volume
                                    inputPhase[b]=1;                            //set flag
                                }                                        
                            }
                        }
                    }else{
                        if((getSW(b+1)!=activeLogicLevel)&&(playing[bufInUse[b]]==0)){         //button released and playback finished                               
                            inputPhase[b]=0;                                                    //get ready for next
                        }
                    }
                    break;
            }
        }
    // check SW4 condition
        switch(SW4mode){
            case SW4TRIGGER:
                SW4rawvalue=0;
                SW4scalevalue=0;            //default to speed value specified at 0%
                break;
            case SW4ANALOG:
                SW4rawvalue=getAnalog();
                if(SW4rawvalue>SW4rawhigh){SW4rawvalue=SW4rawhigh;}     //constrain
                if(SW4rawvalue<SW4rawlow){SW4rawvalue=SW4rawlow;}       //constrain
                if(SW4rawlow==SW4rawhigh){                              //avoid div/0
                    SW4scalevalue=0;
                }else{
                    SW4scalevalue=(SW4_FULL_SCALE*(SW4rawvalue-SW4rawlow))/(SW4rawhigh-SW4rawlow);
                }
                break;
            case SW4FREQUENCY:
                SW4rawvalue=inputFreq;
                if(SW4rawvalue>SW4rawhigh){SW4rawvalue=SW4rawhigh;}     //constrain
                if(SW4rawvalue<SW4rawlow){SW4rawvalue=SW4rawlow;}       //constrain
                if(SW4rawlow==SW4rawhigh){                              //avoid div/0
                    SW4scalevalue=0;
                }else{
                    SW4scalevalue=(SW4_FULL_SCALE*(SW4rawvalue-SW4rawlow))/(SW4rawhigh-SW4rawlow);
                }
                break;
        }
    }
}

void goPower(char p){     //0 for sleep/low, 1 for high/running
    if((pStat!=0)&(p==0)){
        LEDSet(0);    
        IEC1bits.SPI2TXIE=0;                   //disable TX interrupt
        DeinitSPI();                           //shut down card IO
        reg1Set(0);                            //card power
        ampSet(0);                             //amp
        IOIdle();                              //other IO   
        pStat=0;
    }
    if((pStat!=1)&&(p==1)){
        IOInit();
        ampSet(1);
        reg1Set(reg1Needed);
        delay(200);    
        for(i=0;i<10;i++){
            f_mount(0, &fs);                 //mount fs
            stat=auto_mount(&pbuf, &pfs, 0); //inits fs and spi etc
            if(stat==FR_OK){break;}
        }
        if(stat){errorflash(stat);}     //blocking, stalls here on error
        //otherwise, file system is up
        if(!loadConfig("CONFIG.TXT")){errorflash(5);}    //load config file and set operation mode      
        IEC1bits.SPI2TXIE=1;        //enable TX interrupt, needed for any audio output
        pStat=1;
    }
}
